//
//  HomeHeader.swift
//  Do It
//
//  Created by Jim Dovey on 2/11/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI

struct HomeHeader: View {
    static let layout: [[TodoItemGroup]] = [
        [.today, .scheduled],
        [.all, .overdue],
    ]
    
    @Binding var selection: HomeSelection?

    var body: some View {
        VStack {
            ForEach(Self.layout, id: \.self) { row in
                HStack(spacing: 12) {
                    ForEach(row, id: \.self) { group in
                        NavigationLink(destination: TodoList(group: group), tag: .group(group), selection: self.$selection) {
                            HeaderItem(group: group)
                                .hoverEffect(.lift)
                        }
                    }
                }
            }
        }
    }

    private struct HeaderItem: View {
        let group: TodoItemGroup
        @State var itemCount: Int = 0

        @EnvironmentObject var data: DataCenter

        var body: some View {
            VStack(alignment: .leading) {
                HStack {
                    group.icon
                    Spacer()
                    Text("\(itemCount)")
                        .foregroundColor(.primary)
                        .font(.system(.title, design: .rounded))
                        .fontWeight(.bold)
                }
                
                Text(group.title)
                    .foregroundColor(.secondary)
                    .font(.system(.subheadline, design: .rounded))
                    .fontWeight(.medium)
            }
            .onReceive(data.$todoItems) { _ in
                self.itemCount = self.group.items(from: self.data).count
            }
            .padding()
            .background(
                RoundedRectangle(cornerRadius: 15, style: .continuous)
                    .fill(Color(.tertiarySystemBackground))
            )
        }
    }
}

struct HomeHeader_Previews: PreviewProvider {
    static var previews: some View {
        ForEach(ColorScheme.allCases, id: \.self) { colorScheme in
            HomeHeader(selection: .constant(nil))
                .padding()
                .background(Color(.systemGroupedBackground))
                .colorScheme(colorScheme)
        }
        .previewLayout(.sizeThatFits)
        .environmentObject(DataCenter())
    }
}
